import { Router } from 'express';
import { z } from 'zod';
import { Connection } from '@solana/web3.js';
import { nonceStore } from '../stores/nonces';
import { verifySignedMessage } from '../verify';
import { walletOwnsStandardNft } from '../ownership';
import { loadKeypairFromFile, deriveParentNameAccount, ensureSubdomainAndSetUrlRecord } from '../sns';

const rpc = process.env.SOLANA_RPC_URL || 'https://api.mainnet-beta.solana.com';
const connection = new Connection(rpc, 'confirmed');

const parentDomain = process.env.PARENT_DOMAIN || 'opentill.sol';
const keypairPath = process.env.PARENT_AUTHORITY_KEYPAIR_PATH || '';
// Base URL for role paths, ex: https://mint.opentill.io
const profileBaseUrl = process.env.PROFILE_BASE_URL || 'https://mint.opentill.io';

export const subdomainRouter = Router();

const Body = z.object({
  wallet: z.string().min(32),
  phase: z.enum(['AMBASSADOR', 'MERCHANT', 'CONSUMER']),
  nftMint: z.string().min(32),
  subdomain: z.string().min(3),
  nonce: z.string().min(8),
  message: z.string().min(10),
  signature: z.string().min(20), // base64
});

function expectedSubdomain(phase: string, nftMint: string) {
  const role = phase === 'AMBASSADOR' ? 'amb' : phase === 'MERCHANT' ? 'merch' : 'cons';
  const mintPrefix = nftMint.replace(/[^A-Za-z0-9]/g, '').slice(0, 6);
  return `${role}-${mintPrefix}.${parentDomain}`;
}

function rolePath(phase: string) {
  if (phase === 'AMBASSADOR') return 'ambassador';
  if (phase === 'MERCHANT') return 'merchant';
  return 'consumer';
}

function buildRoleUrl(phase: string, nftMint: string) {
  const base = profileBaseUrl.replace(/\/+$/, '');
  return `${base}/${rolePath(phase)}/${nftMint}`;
}

subdomainRouter.post('/create', async (req, res) => {
  try {
    const parsed = Body.safeParse(req.body);
    if (!parsed.success) return res.status(400).send(parsed.error.message);

    const { wallet, phase, nftMint, subdomain, nonce, message, signature } = parsed.data;

    nonceStore.consume({ nonce, action: 'CREATE', wallet });
    verifySignedMessage({ wallet, message, signatureB64: signature });

    const expMsg = `openTILL create: ${subdomain} for mint ${nftMint} nonce ${nonce}`;
    if (message !== expMsg) throw new Error('Message mismatch');

    const expSub = expectedSubdomain(phase, nftMint);
    if (subdomain !== expSub) throw new Error(`Invalid subdomain. Expected ${expSub}`);

    const owns = await walletOwnsStandardNft({ connection, wallet, nftMint });
    if (!owns) throw new Error('Wallet does not currently own this NFT mint.');

    if (!keypairPath) throw new Error('Server missing PARENT_AUTHORITY_KEYPAIR_PATH');
    const payer = loadKeypairFromFile(keypairPath);

    const parentNameAccount = await deriveParentNameAccount(parentDomain);

    const subLabel = subdomain.split(`.${parentDomain}`)[0];
    const url = buildRoleUrl(phase, nftMint);

    await ensureSubdomainAndSetUrlRecord({
      connection,
      payer,
      parentDomain,
      parentNameAccount,
      subLabel,
      url,
    });

    res.json({ subdomain });
  } catch (e: any) {
    res.status(400).send(e?.message || 'Create failed');
  }
});

subdomainRouter.post('/claim', async (req, res) => {
  try {
    const parsed = Body.safeParse(req.body);
    if (!parsed.success) return res.status(400).send(parsed.error.message);

    const { wallet, phase, nftMint, subdomain, nonce, message, signature } = parsed.data;

    nonceStore.consume({ nonce, action: 'CLAIM', wallet });
    verifySignedMessage({ wallet, message, signatureB64: signature });

    const expMsg = `openTILL claim: ${subdomain} for mint ${nftMint} nonce ${nonce}`;
    if (message !== expMsg) throw new Error('Message mismatch');

    const expSub = expectedSubdomain(phase, nftMint);
    if (subdomain !== expSub) throw new Error(`Invalid subdomain. Expected ${expSub}`);

    const owns = await walletOwnsStandardNft({ connection, wallet, nftMint });
    if (!owns) throw new Error('Wallet does not currently own this NFT mint.');

    if (!keypairPath) throw new Error('Server missing PARENT_AUTHORITY_KEYPAIR_PATH');
    const payer = loadKeypairFromFile(keypairPath);

    const parentNameAccount = await deriveParentNameAccount(parentDomain);

    const subLabel = subdomain.split(`.${parentDomain}`)[0];
    const url = buildRoleUrl(phase, nftMint);

    await ensureSubdomainAndSetUrlRecord({
      connection,
      payer,
      parentDomain,
      parentNameAccount,
      subLabel,
      url,
    });

    res.json({ subdomain });
  } catch (e: any) {
    res.status(400).send(e?.message || 'Claim failed');
  }
});
