/**
 * Solana Name Service (SNS) helpers
 *
 * This implements:
 * - Deriving the parent domain name account (opentill.sol)
 * - Creating a subdomain (merch-xxxxxx.opentill.sol) if missing
 * - Creating/updating a V1 `url` record for that subdomain
 *
 * Why V1 for URL?
 * - It is widely supported and is what many resolvers check first (URL record).
 * - V2 exists, but V1 URL record remains a common compatibility target.
 *
 * Sources (docs):
 * - Subdomains: createSubdomain(connection, subdomain, owner)
 * - Records V1: createNameRegistry + updateInstruction for Record.Url
 */
import fs from 'fs';
import path from 'path';
import { Connection, Keypair, PublicKey, Transaction, TransactionInstruction } from '@solana/web3.js';
import {
  Record,
  getDomainKey,
  createSubdomain,
  createNameRegistry,
  NameRegistryState,
  updateInstruction,
  NAME_PROGRAM_ID,
  Numberu32,
} from '@bonfida/spl-name-service';

export function loadKeypairFromFile(filePath: string): Keypair {
  const absolute = path.resolve(filePath);
  const raw = fs.readFileSync(absolute, 'utf8');
  const secret = Uint8Array.from(JSON.parse(raw));
  return Keypair.fromSecretKey(secret);
}

export async function deriveParentNameAccount(parentDomain: string): Promise<PublicKey> {
  // parentDomain can be with or without .sol. We'll pass as-is; SDK handles both.
  const { pubkey } = await getDomainKey(parentDomain);
  return pubkey;
}

async function ensureSubdomain(params: {
  connection: Connection;
  payer: Keypair;
  parentDomain: string; // e.g. opentill.sol
  subLabel: string; // e.g. merch-9f3a12
}): Promise<string> {
  const { connection, payer, parentDomain, subLabel } = params;
  const full = `${subLabel}.${parentDomain}`;

  const { pubkey } = await getDomainKey(full);
  const existing = await connection.getAccountInfo(pubkey);
  if (existing) return full;

  const ix = createSubdomain(connection, full, payer.publicKey);
  const tx = new Transaction().add(ix);
  const sig = await connection.sendTransaction(tx, [payer], { skipPreflight: false });
  await connection.confirmTransaction(sig, 'confirmed');

  return full;
}

async function ensureAndSetUrlRecordV1(params: {
  connection: Connection;
  payer: Keypair;
  domain: string; // with or without .sol
  url: string;
}) {
  const { connection, payer, domain, url } = params;

  const ixs: TransactionInstruction[] = [];

  // Domain key of the (sub)domain itself
  const { pubkey: domainKey } = await getDomainKey(domain);

  // Record key for url record
  const record = Record.Url; // "url"
  const { pubkey: recordKey } = await getDomainKey(record + "." + domain, true);

  const recordAccInfo = await connection.getAccountInfo(recordKey);
  if (!recordAccInfo?.data) {
    // Create record registry first (V1 prefix is 0x01)
    const space = 2000;
    const lamports = await connection.getMinimumBalanceForRentExemption(space + NameRegistryState.HEADER_LEN);

    const createIx = await createNameRegistry(
      connection,
      Buffer.from([1]).toString() + record, // V1 derivation prefix
      space,
      payer.publicKey,
      payer.publicKey,
      lamports,
      undefined,
      domainKey
    );
    ixs.push(createIx);
  }

  const updateIx = updateInstruction(
    NAME_PROGRAM_ID,
    recordKey,
    new Numberu32(0),
    Buffer.from(url),
    payer.publicKey
  );

  ixs.push(updateIx);

  const tx = new Transaction().add(...ixs);
  const sig = await connection.sendTransaction(tx, [payer], { skipPreflight: false });
  await connection.confirmTransaction(sig, 'confirmed');
}

export async function ensureSubdomainAndSetUrlRecord(params: {
  connection: Connection;
  payer: Keypair;
  parentDomain: string;
  parentNameAccount: PublicKey; // derived, but included for clarity/logging
  subLabel: string;
  url: string;
}): Promise<void> {
  const { connection, payer, parentDomain, subLabel, url } = params;

  // Ensure subdomain exists
  const subdomain = await ensureSubdomain({ connection, payer, parentDomain, subLabel });

  // Ensure URL record exists and is updated
  await ensureAndSetUrlRecordV1({ connection, payer, domain: subdomain, url });
}
