import React,{useState} from 'react';
import {WalletMultiButton} from '@solana/wallet-adapter-react-ui';
import {useWallet} from '@solana/wallet-adapter-react';
import {CONFIG, Phase} from './config';
import {mintFromCandyMachine, explorerTx} from './lib/mintCandyMachine';
import {createSubdomain, claimSubdomain, subdomainForMint} from './lib/opentillName';
import {Card,CardBody,CardHeader} from './components/Card';
import {Button} from './components/Button';
import {Badge} from './components/Badge';

const PHASES:Phase[]=['AMBASSADOR','MERCHANT','CONSUMER'];
function shortPk(pk:string){ if(!pk||pk.includes('REPLACE')) return pk; return pk.slice(0,4)+'…'+pk.slice(-4); }

export default function App(){
  const wallet=useWallet();
  const [phase,setPhase]=useState<Phase>('AMBASSADOR');
  const [busy,setBusy]=useState(false);
  const [nameBusy,setNameBusy]=useState(false);
  const [tx,setTx]=useState<string|null>(null);
  const [nftMint,setNftMint]=useState<string|null>(null);
  const [subdomain,setSubdomain]=useState<string|null>(null);
  const [msg,setMsg]=useState<string|null>(null);
  const [error,setError]=useState<string|null>(null);

  const candyMachineId=CONFIG.candyMachineId[phase];
  const canMint=!!wallet.connected && !!wallet.publicKey && !busy;

  async function onMint(){
    try{
      setError(null); setMsg(null); setTx(null); setNftMint(null); setSubdomain(null);
      setBusy(true);
      if(!candyMachineId || candyMachineId.includes('REPLACE')) throw new Error('Candy Machine ID not set.');
      const res=await mintFromCandyMachine({rpcUrl:CONFIG.rpcUrl,candyMachineId,wallet});
      setTx(res.signature);
      setNftMint(res.nftMint);

      setNameBusy(true);
      const created=await createSubdomain({wallet,phase,nftMint:res.nftMint});
      setSubdomain(created.subdomain);
      setMsg('Subdomain provisioned. If you sell the NFT, the new owner can claim control.');
    }catch(e:any){
      setError(e?.message||'Mint failed.');
    }finally{
      setBusy(false); setNameBusy(false);
    }
  }

  async function onClaim(){
    try{
      setError(null); setMsg(null);
      if(!nftMint) throw new Error('No NFT mint in session.');
      setNameBusy(true);
      const claimed=await claimSubdomain({wallet,phase,nftMint});
      setSubdomain(claimed.subdomain);
      setMsg('Claim updated. You control the subdomain records.');
    }catch(e:any){
      setError(e?.message||'Claim failed.');
    }finally{
      setNameBusy(false);
    }
  }

  const displayedSubdomain=subdomain || (nftMint ? subdomainForMint(phase,nftMint) : null);

  return (
    <div className="min-h-screen">
      <header className="sticky top-0 z-20 border-b border-zinc-900 bg-zinc-950/80 backdrop-blur">
        <div className="mx-auto max-w-6xl px-4 py-3 flex items-center justify-between gap-3">
          <div className="flex items-center gap-3">
            <div className="h-9 w-9 rounded-xl bg-zinc-900 border border-zinc-800 grid place-items-center font-black">OT</div>
            <div className="leading-tight">
              <div className="text-sm font-semibold">openTILL Identity Mint</div>
              <div className="text-xs text-zinc-400">SMPF Roles: Ambassador • Merchant • Consumer</div>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <Badge>Free mint (tx fees apply)</Badge>
            <WalletMultiButton />
          </div>
        </div>
      </header>

      <main className="mx-auto max-w-6xl px-4 py-10 grid gap-8">
        <section className="grid gap-4">
          <h1 className="text-3xl sm:text-4xl font-black tracking-tight">Role NFTs + NFT-bound opentill.sol identity</h1>
          <p className="text-zinc-300 max-w-3xl">
            Mint a tradeable identity NFT and receive an NFT-bound SNS subdomain under <span className="text-zinc-100 font-semibold">opentill.sol</span>.
            Subdomains are deterministic and tied to the NFT mint. Holders claim record control with a signed message.
          </p>
          <div className="flex flex-wrap gap-2">
            <Badge>Delayed reveal</Badge><Badge>Role-locked multipliers</Badge><Badge>One-per-role cap</Badge><Badge>5% royalties</Badge><Badge>opentill.sol</Badge>
          </div>
        </section>

        <section className="grid lg:grid-cols-3 gap-6">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between gap-3">
                <div>
                  <div className="text-sm text-zinc-400">Select phase</div>
                  <div className="text-lg font-bold">{CONFIG.phaseSchedule[phase].label}</div>
                </div>
                <Badge>CM: {shortPk(candyMachineId)}</Badge>
              </div>
            </CardHeader>
            <CardBody>
              <div className="grid gap-3">
                <div className="flex flex-wrap gap-2">
                  {PHASES.map(p => (
                    <Button key={p} variant={p===phase?'primary':'ghost'} onClick={()=>setPhase(p)} className="w-full sm:w-auto">
                      {CONFIG.phaseSchedule[p].label}
                    </Button>
                  ))}
                </div>

                <div className="grid gap-2">
                  <Button onClick={onMint} disabled={!canMint}>{busy?'Minting…':'Mint 1 NFT'}</Button>
                  {tx && (
                    <div className="text-sm">
                      <div className="text-zinc-300">Mint confirmed.</div>
                      <a className="text-zinc-100 underline" href={explorerTx(tx)} target="_blank" rel="noreferrer">View transaction</a>
                    </div>
                  )}
                  {nftMint && <div className="text-xs text-zinc-400">Mint: <span className="text-zinc-200">{nftMint}</span></div>}

                  {displayedSubdomain && (
                    <div className="rounded-2xl border border-zinc-900 bg-zinc-950 p-4 grid gap-2">
                      <div className="text-xs text-zinc-500">Your NFT-bound name</div>
                      <div className="text-sm font-semibold">{displayedSubdomain}</div>
                      <div className="text-xs text-zinc-400">Bound to the NFT mint. New owners can claim control after resale.</div>
                      <Button onClick={onClaim} variant="ghost" disabled={!wallet.connected || !nftMint || nameBusy}>
                        {nameBusy?'Working…':'Claim / Refresh control'}
                      </Button>
                      {msg && <div className="text-xs text-zinc-300">{msg}</div>}
                    </div>
                  )}

                  {error && <div className="text-sm text-red-300 border border-red-900/50 bg-red-950/30 rounded-xl p-3">{error}</div>}
                </div>
              </div>
            </CardBody>
          </Card>

          <Card className="lg:col-span-2">
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <div className="text-sm text-zinc-400">Utility rules</div>
                  <div className="text-lg font-bold">Role-locked + permanent</div>
                </div>
                <Badge>Collection: {shortPk(CONFIG.collectionAddress)}</Badge>
              </div>
            </CardHeader>
            <CardBody>
              <div className="grid sm:grid-cols-3 gap-4">
                <div className="rounded-2xl border border-zinc-900 bg-zinc-950 p-4">
                  <div className="font-semibold">Ambassador</div>
                  <ul className="mt-3 text-xs text-zinc-400 list-disc pl-4 space-y-1"><li>Common 1.00x</li><li>Rare 1.10x</li><li>Epic 1.25x</li><li>Legendary 1.50x</li></ul>
                </div>
                <div className="rounded-2xl border border-zinc-900 bg-zinc-950 p-4">
                  <div className="font-semibold">Merchant</div>
                  <ul className="mt-3 text-xs text-zinc-400 list-disc pl-4 space-y-1"><li>Common 1.00x</li><li>Rare 1.10x</li><li>Epic 1.25x</li><li>Legendary 1.50x</li></ul>
                </div>
                <div className="rounded-2xl border border-zinc-900 bg-zinc-950 p-4">
                  <div className="font-semibold">Consumer</div>
                  <ul className="mt-3 text-xs text-zinc-400 list-disc pl-4 space-y-1"><li>Common 1.00x</li><li>Rare 1.05x</li><li>Epic 1.15x</li><li>Legendary 1.25x</li></ul>
                </div>
              </div>
              <div className="mt-6 rounded-2xl border border-zinc-900 bg-zinc-950 p-4">
                <div className="font-semibold">How the NFT-bound name works</div>
                <p className="mt-2 text-sm text-zinc-300">
                  After mint, the app provisions <span className="text-zinc-100 font-semibold">role-mintPrefix.opentill.sol</span>.
                  The server stores <span className="text-zinc-100 font-semibold">nft_mint</span> + <span className="text-zinc-100 font-semibold">controller</span> in SNS records.
                  Whoever currently owns the NFT can claim control by signing a message.
                </p>
              </div>
            </CardBody>
          </Card>
        </section>

        <footer className="py-8 text-xs text-zinc-500">© {new Date().getFullYear()} openTILL.</footer>
      </main>
    </div>
  );
}
